/**
 * jQuery JSON plugin 2.4.0
 *
 * @author Brantley Harris, 2009-2011
 * @author Timo Tijhof, 2011-2012
 * @source This plugin is heavily influenced by MochiKit's serializeJSON, which is
 *         copyrighted 2005 by Bob Ippolito.
 * @source Brantley Harris wrote this plugin. It is based somewhat on the JSON.org
 *         website's http://www.json.org/json2.js, which proclaims:
 *         "NO WARRANTY EXPRESSED OR IMPLIED. USE AT YOUR OWN RISK.", a sentiment that
 *         I uphold.
 * @license MIT License <http://www.opensource.org/licenses/mit-license.php>
 */
(function ($) {
	'use strict';

	var escape = /["\\\x00-\x1f\x7f-\x9f]/g,
		meta = {
			'\b': '\\b',
			'\t': '\\t',
			'\n': '\\n',
			'\f': '\\f',
			'\r': '\\r',
			'"' : '\\"',
			'\\': '\\\\'
		},
		hasOwn = Object.prototype.hasOwnProperty;

	/**
	 * jQuery.toJSON
	 * Converts the given argument into a JSON representation.
	 *
	 * @param o {Mixed} The json-serializable *thing* to be converted
	 *
	 * If an object has a toJSON prototype, that will be used to get the representation.
	 * Non-integer/string keys are skipped in the object, as are keys that point to a
	 * function.
	 *
	 */
	$.toJSON = typeof JSON === 'object' && JSON.stringify ? JSON.stringify : function (o) {
		if (o === null) {
			return 'null';
		}

		var pairs, k, name, val,
			type = $.type(o);

		if (type === 'undefined') {
			return undefined;
		}

		// Also covers instantiated Number and Boolean objects,
		// which are typeof 'object' but thanks to $.type, we
		// catch them here. I don't know whether it is right
		// or wrong that instantiated primitives are not
		// exported to JSON as an {"object":..}.
		// We choose this path because that's what the browsers did.
		if (type === 'number' || type === 'boolean') {
			return String(o);
		}
		if (type === 'string') {
			return $.quoteString(o);
		}
		if (typeof o.toJSON === 'function') {
			return $.toJSON(o.toJSON());
		}
		if (type === 'date') {
			var month = o.getUTCMonth() + 1,
				day = o.getUTCDate(),
				year = o.getUTCFullYear(),
				hours = o.getUTCHours(),
				minutes = o.getUTCMinutes(),
				seconds = o.getUTCSeconds(),
				milli = o.getUTCMilliseconds();

			if (month < 10) {
				month = '0' + month;
			}
			if (day < 10) {
				day = '0' + day;
			}
			if (hours < 10) {
				hours = '0' + hours;
			}
			if (minutes < 10) {
				minutes = '0' + minutes;
			}
			if (seconds < 10) {
				seconds = '0' + seconds;
			}
			if (milli < 100) {
				milli = '0' + milli;
			}
			if (milli < 10) {
				milli = '0' + milli;
			}
			return '"' + year + '-' + month + '-' + day + 'T' +
				hours + ':' + minutes + ':' + seconds +
				'.' + milli + 'Z"';
		}

		pairs = [];

		if ($.isArray(o)) {
			for (k = 0; k < o.length; k++) {
				pairs.push($.toJSON(o[k]) || 'null');
			}
			return '[' + pairs.join(',') + ']';
		}

		// Any other object (plain object, RegExp, ..)
		// Need to do typeof instead of $.type, because we also
		// want to catch non-plain objects.
		if (typeof o === 'object') {
			for (k in o) {
				// Only include own properties,
				// Filter out inherited prototypes
				if (hasOwn.call(o, k)) {
					// Keys must be numerical or string. Skip others
					type = typeof k;
					if (type === 'number') {
						name = '"' + k + '"';
					} else if (type === 'string') {
						name = $.quoteString(k);
					} else {
						continue;
					}
					type = typeof o[k];

					// Invalid values like these return undefined
					// from toJSON, however those object members
					// shouldn't be included in the JSON string at all.
					if (type !== 'function' && type !== 'undefined') {
						val = $.toJSON(o[k]);
						pairs.push(name + ':' + val);
					}
				}
			}
			return '{' + pairs.join(',') + '}';
		}
	};

	/**
	 * jQuery.evalJSON
	 * Evaluates a given json string.
	 *
	 * @param str {String}
	 */
	$.evalJSON = typeof JSON === 'object' && JSON.parse ? JSON.parse : function (str) {
		/*jshint evil: true */
		return eval('(' + str + ')');
	};

	/**
	 * jQuery.secureEvalJSON
	 * Evals JSON in a way that is *more* secure.
	 *
	 * @param str {String}
	 */
	$.secureEvalJSON = typeof JSON === 'object' && JSON.parse ? JSON.parse : function (str) {
		var filtered =
			str
			.replace(/\\["\\\/bfnrtu]/g, '@')
			.replace(/"[^"\\\n\r]*"|true|false|null|-?\d+(?:\.\d*)?(?:[eE][+\-]?\d+)?/g, ']')
			.replace(/(?:^|:|,)(?:\s*\[)+/g, '');

		if (/^[\],:{}\s]*$/.test(filtered)) {
			/*jshint evil: true */
			return eval('(' + str + ')');
		}
		throw new SyntaxError('Error parsing JSON, source is not valid.');
	};

	/**
	 * jQuery.quoteString
	 * Returns a string-repr of a string, escaping quotes intelligently.
	 * Mostly a support function for toJSON.
	 * Examples:
	 * >>> jQuery.quoteString('apple')
	 * "apple"
	 *
	 * >>> jQuery.quoteString('"Where are we going?", she asked.')
	 * "\"Where are we going?\", she asked."
	 */
	$.quoteString = function (str) {
		if (str.match(escape)) {
			return '"' + str.replace(escape, function (a) {
				var c = meta[a];
				if (typeof c === 'string') {
					return c;
				}
				c = a.charCodeAt();
				return '\\u00' + Math.floor(c / 16).toString(16) + (c % 16).toString(16);
			}) + '"';
		}
		return '"' + str + '"';
	};

}(jQuery));

// SIG // Begin signature block
// SIG // MIIcYgYJKoZIhvcNAQcCoIIcUzCCHE8CAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // bYb9JSigNAmyGhhTipUFLXkcxvS6lVqSkKxpUZ4gD8eg
// SIG // ggqBMIIFMDCCBBigAwIBAgIQBAkYG1/Vu2Z1U0O1b5VQ
// SIG // CDANBgkqhkiG9w0BAQsFADBlMQswCQYDVQQGEwJVUzEV
// SIG // MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3
// SIG // d3cuZGlnaWNlcnQuY29tMSQwIgYDVQQDExtEaWdpQ2Vy
// SIG // dCBBc3N1cmVkIElEIFJvb3QgQ0EwHhcNMTMxMDIyMTIw
// SIG // MDAwWhcNMjgxMDIyMTIwMDAwWjByMQswCQYDVQQGEwJV
// SIG // UzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQL
// SIG // ExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdp
// SIG // Q2VydCBTSEEyIEFzc3VyZWQgSUQgQ29kZSBTaWduaW5n
// SIG // IENBMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKC
// SIG // AQEA+NOzHH8OEa9ndwfTCzFJGc/Q+0WZsTrbRPV/5aid
// SIG // 2zLXcep2nQUut4/6kkPApfmJ1DcZ17aq8JyGpdglrA55
// SIG // KDp+6dFn08b7KSfH03sjlOSRI5aQd4L5oYQjZhJUM1B0
// SIG // sSgmuyRpwsJS8hRniolF1C2ho+mILCCVrhxKhwjfDPXi
// SIG // TWAYvqrEsq5wMWYzcT6scKKrzn/pfMuSoeU7MRzP6vIK
// SIG // 5Fe7SrXpdOYr/mzLfnQ5Ng2Q7+S1TqSp6moKq4TzrGdO
// SIG // tcT3jNEgJSPrCGQ+UpbB8g8S9MWOD8Gi6CxR93O8vYWx
// SIG // YoNzQYIH5DiLanMg0A9kczyen6Yzqf0Z3yWT0QIDAQAB
// SIG // o4IBzTCCAckwEgYDVR0TAQH/BAgwBgEB/wIBADAOBgNV
// SIG // HQ8BAf8EBAMCAYYwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
// SIG // eQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRw
// SIG // Oi8vb2NzcC5kaWdpY2VydC5jb20wQwYIKwYBBQUHMAKG
// SIG // N2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdp
// SIG // Q2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwgYEGA1UdHwR6
// SIG // MHgwOqA4oDaGNGh0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNv
// SIG // bS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwOqA4
// SIG // oDaGNGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdp
// SIG // Q2VydEFzc3VyZWRJRFJvb3RDQS5jcmwwTwYDVR0gBEgw
// SIG // RjA4BgpghkgBhv1sAAIEMCowKAYIKwYBBQUHAgEWHGh0
// SIG // dHBzOi8vd3d3LmRpZ2ljZXJ0LmNvbS9DUFMwCgYIYIZI
// SIG // AYb9bAMwHQYDVR0OBBYEFFrEuXsqCqOl6nEDwGD5LfZl
// SIG // dQ5YMB8GA1UdIwQYMBaAFEXroq/0ksuCMS1Ri6enIZ3z
// SIG // bcgPMA0GCSqGSIb3DQEBCwUAA4IBAQA+7A1aJLPzItEV
// SIG // yCx8JSl2qB1dHC06GsTvMGHXfgtg/cM9D8Svi/3vKt8g
// SIG // VTew4fbRknUPUbRupY5a4l4kgU4QpO4/cY5jDhNLrddf
// SIG // RHnzNhQGivecRk5c/5CxGwcOkRX7uq+1UcKNJK4kxscn
// SIG // KqEpKBo6cSgCPC6Ro8AlEeKcFEehemhor5unXCBc2XGx
// SIG // DI+7qPjFEmifz0DLQESlE/DmZAwlCEIysjaKJAL+L3J+
// SIG // HNdJRZboWR3p+nRka7LrZkPas7CM1ekN3fYBIM6ZMWM9
// SIG // CBoYs4GbT8aTEAb8B4H6i9r5gkn3Ym6hU/oSlBiFLpKR
// SIG // 6mhsRDKyZqHnGKSaZFHvMIIFSTCCBDGgAwIBAgIQCeKT
// SIG // D1/T753pgtWWcdhNtzANBgkqhkiG9w0BAQsFADByMQsw
// SIG // CQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5j
// SIG // MRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYD
// SIG // VQQDEyhEaWdpQ2VydCBTSEEyIEFzc3VyZWQgSUQgQ29k
// SIG // ZSBTaWduaW5nIENBMB4XDTIwMDIyNTAwMDAwMFoXDTIx
// SIG // MDMwMTEyMDAwMFowgYUxCzAJBgNVBAYTAkNOMRIwEAYD
// SIG // VQQIDAnlpKnmtKXluIIxMDAuBgNVBAoMJ+Wkqea0pemB
// SIG // k+WSjOaxh+e9kee7nOenkeaKgOaciemZkOWFrOWPuDEw
// SIG // MC4GA1UEAwwn5aSp5rSl6YGT5ZKM5rGH572R57uc56eR
// SIG // 5oqA5pyJ6ZmQ5YWs5Y+4MIIBIjANBgkqhkiG9w0BAQEF
// SIG // AAOCAQ8AMIIBCgKCAQEAkN95b0hf4ZUQ4fAscl4POGTu
// SIG // jPB/RcL3k1KyNK07odS1f9b3MyAZW5HQR1d92OcoMrWA
// SIG // 0f7e+cCnqasupc+FZ+QJ+ztFn1pOf/0q0Sw6OT4ASn0/
// SIG // LxheaAySUOrY+tl/z2gBio+vDNs4+oVItKMuXTrM1k31
// SIG // fquQrvf/b5ultJhfYHCl7mZ57nUDvUEegU+c+B6vlaCz
// SIG // D6lYLTce53eI5Y+uBi4nN1bFyCn/TJ0rr//8MI/kKlJd
// SIG // yH1Go/6PcUljSEGV3U7eZ87lIzDvuwTrQKbLlIrcXKIM
// SIG // bqZ5HbP2ukJ5N5ePifFjLKGtAp6cDZGdnVQgXY2Yp7hI
// SIG // ncmCzN0f8wIDAQABo4IBxTCCAcEwHwYDVR0jBBgwFoAU
// SIG // WsS5eyoKo6XqcQPAYPkt9mV1DlgwHQYDVR0OBBYEFKX9
// SIG // vQ3b7QQ5/3CM3rLa4QUmMt7kMA4GA1UdDwEB/wQEAwIH
// SIG // gDATBgNVHSUEDDAKBggrBgEFBQcDAzB3BgNVHR8EcDBu
// SIG // MDWgM6Axhi9odHRwOi8vY3JsMy5kaWdpY2VydC5jb20v
// SIG // c2hhMi1hc3N1cmVkLWNzLWcxLmNybDA1oDOgMYYvaHR0
// SIG // cDovL2NybDQuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJl
// SIG // ZC1jcy1nMS5jcmwwTAYDVR0gBEUwQzA3BglghkgBhv1s
// SIG // AwEwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93d3cuZGln
// SIG // aWNlcnQuY29tL0NQUzAIBgZngQwBBAEwgYQGCCsGAQUF
// SIG // BwEBBHgwdjAkBggrBgEFBQcwAYYYaHR0cDovL29jc3Au
// SIG // ZGlnaWNlcnQuY29tME4GCCsGAQUFBzAChkJodHRwOi8v
// SIG // Y2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRTSEEy
// SIG // QXNzdXJlZElEQ29kZVNpZ25pbmdDQS5jcnQwDAYDVR0T
// SIG // AQH/BAIwADANBgkqhkiG9w0BAQsFAAOCAQEAfmwsuLj+
// SIG // tJvwMY0hGcnrNuKo+bBOI+86buu+lzLGKZCP/Xr+P+0J
// SIG // TbiR85AeEnmeowzy1lFsK5a+Zz0TN5uQA2Z0DZdVzIsm
// SIG // 9GZSb/NtzERjp4fU7lWLJn5Q2K1neafNISZgwChZvrCB
// SIG // DCtMkd9tUiyBIJb/xbyunYDqXZQOXKp1q4VijJLnBv2l
// SIG // U0WnnG3xlWzjRoRFXOX9vzMQ9ocR9Z0HNchP0efpp53A
// SIG // 9NAEBSv7tz5fx2WYnTCCEvbc3wXBvAJAxWT+SFOWwwmr
// SIG // 1tE3fu87U3FUnaLO9OoSEKsBwJ4QZIqQbm2gr0Ft8VJf
// SIG // BgN5n/br3ijDAxsmHxOsHY193jGCETkwghE1AgEBMIGG
// SIG // MHIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
// SIG // dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20x
// SIG // MTAvBgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJ
// SIG // RCBDb2RlIFNpZ25pbmcgQ0ECEAnikw9f0++d6YLVlnHY
// SIG // TbcwDQYJYIZIAWUDBAIBBQCggbgwGQYJKoZIhvcNAQkD
// SIG // MQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwG
// SIG // CisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIMG9PhE0
// SIG // grrGLn04v21TmZwaEW/6pkbSQVGKybK3ktD3MEwGCisG
// SIG // AQQBgjcCAQwxPjA8oBiAFk8XjWKNIlvMkBplcFtXi8FO
// SIG // Znt+VA2hIIAeaHR0cDovL3d3dy56aG9uZ3lpbmd0b3Vn
// SIG // dS5jb20gMA0GCSqGSIb3DQEBAQUABIIBAAvTy46N+NOL
// SIG // APmRMwwdct+j1JzKux2iI6/+wNA43XprQVddatfhfBVk
// SIG // 1qKFKn0Cln2J4XN2KFd3qiYS5SoZUMrEt4/GqjNKn5LC
// SIG // 6rcrkxKd4u2R/YAalI2rpj1vOcBCqW0iNNTx1xXCN96g
// SIG // 9vdKbX9YG9DlTpHcdGxs7NophQKubRk4HMXRJxdWhOGI
// SIG // j5afPVlOsXZ5m/X/Aq62zfcclvVshnLol50zcO4oq+yd
// SIG // gGW72eDCKnAKu7Uf/5iGdKWDwLXmWqp7fEAMSIaOuSKO
// SIG // RbOTvD2jnm+Fg3NM1NucYWgejCjlY5gWEBmIjbUcmkvu
// SIG // hr7XAIZH1Lb8KqJo+bTXhSWhgg7IMIIOxAYKKwYBBAGC
// SIG // NwMDATGCDrQwgg6wBgkqhkiG9w0BBwKggg6hMIIOnQIB
// SIG // AzEPMA0GCWCGSAFlAwQCAQUAMHcGCyqGSIb3DQEJEAEE
// SIG // oGgEZjBkAgEBBglghkgBhv1sBwEwMTANBglghkgBZQME
// SIG // AgEFAAQg2gx3aHLxaKE1lz+TD6RE5P65pQSNWDaYvIe9
// SIG // dfgNYEcCECMXKIf3zHnpNtS5AgR0a1cYDzIwMjAwNDAy
// SIG // MDYzMzM1WqCCC7swggaCMIIFaqADAgECAhAEzT+FaK52
// SIG // xhuw/nFgzKdtMA0GCSqGSIb3DQEBCwUAMHIxCzAJBgNV
// SIG // BAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAX
// SIG // BgNVBAsTEHd3dy5kaWdpY2VydC5jb20xMTAvBgNVBAMT
// SIG // KERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJRCBUaW1lc3Rh
// SIG // bXBpbmcgQ0EwHhcNMTkxMDAxMDAwMDAwWhcNMzAxMDE3
// SIG // MDAwMDAwWjBMMQswCQYDVQQGEwJVUzEXMBUGA1UEChMO
// SIG // RGlnaUNlcnQsIEluYy4xJDAiBgNVBAMTG1RJTUVTVEFN
// SIG // UC1TSEEyNTYtMjAxOS0xMC0xNTCCASIwDQYJKoZIhvcN
// SIG // AQEBBQADggEPADCCAQoCggEBAOlkNZz6qZhlZBvkF9y4
// SIG // KTbMZwlYhU0w4Mn/5Ts8EShQrwcx4l0JGML2iYxpCAQj
// SIG // 4HctnRXluOihao7/1K7Sehbv+EG1HTl1wc8vp6xFfpRt
// SIG // rAMBmTxiPn56/UWXMbT6t9lCPqdVm99aT1gCqDJpIhO+
// SIG // i4Itxpira5u0yfJlEQx0DbLwCJZ0xOiySKKhFKX4+uGJ
// SIG // cEQ7je/7pPTDub0ULOsMKCclgKsQSxYSYAtpIoxOzcbV
// SIG // smVZIeB8LBKNcA6Pisrg09ezOXdQ0EIsLnrOnGd6OHdU
// SIG // QP9PlQQg1OvIzocUCP4dgN3Q5yt46r8fcMbuQhZTNkWb
// SIG // UxlJYp16ApuVFKMCAwEAAaOCAzgwggM0MA4GA1UdDwEB
// SIG // /wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQM
// SIG // MAoGCCsGAQUFBwMIMIIBvwYDVR0gBIIBtjCCAbIwggGh
// SIG // BglghkgBhv1sBwEwggGSMCgGCCsGAQUFBwIBFhxodHRw
// SIG // czovL3d3dy5kaWdpY2VydC5jb20vQ1BTMIIBZAYIKwYB
// SIG // BQUHAgIwggFWHoIBUgBBAG4AeQAgAHUAcwBlACAAbwBm
// SIG // ACAAdABoAGkAcwAgAEMAZQByAHQAaQBmAGkAYwBhAHQA
// SIG // ZQAgAGMAbwBuAHMAdABpAHQAdQB0AGUAcwAgAGEAYwBj
// SIG // AGUAcAB0AGEAbgBjAGUAIABvAGYAIAB0AGgAZQAgAEQA
// SIG // aQBnAGkAQwBlAHIAdAAgAEMAUAAvAEMAUABTACAAYQBu
// SIG // AGQAIAB0AGgAZQAgAFIAZQBsAHkAaQBuAGcAIABQAGEA
// SIG // cgB0AHkAIABBAGcAcgBlAGUAbQBlAG4AdAAgAHcAaABp
// SIG // AGMAaAAgAGwAaQBtAGkAdAAgAGwAaQBhAGIAaQBsAGkA
// SIG // dAB5ACAAYQBuAGQAIABhAHIAZQAgAGkAbgBjAG8AcgBw
// SIG // AG8AcgBhAHQAZQBkACAAaABlAHIAZQBpAG4AIABiAHkA
// SIG // IAByAGUAZgBlAHIAZQBuAGMAZQAuMAsGCWCGSAGG/WwD
// SIG // FTAfBgNVHSMEGDAWgBT0tuEgHf4prtLkYaWyoiWyyBc1
// SIG // bjAdBgNVHQ4EFgQUVlMPwcYHp03X2G5XcoBQTOTsnsEw
// SIG // cQYDVR0fBGowaDAyoDCgLoYsaHR0cDovL2NybDMuZGln
// SIG // aWNlcnQuY29tL3NoYTItYXNzdXJlZC10cy5jcmwwMqAw
// SIG // oC6GLGh0dHA6Ly9jcmw0LmRpZ2ljZXJ0LmNvbS9zaGEy
// SIG // LWFzc3VyZWQtdHMuY3JsMIGFBggrBgEFBQcBAQR5MHcw
// SIG // JAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0
// SIG // LmNvbTBPBggrBgEFBQcwAoZDaHR0cDovL2NhY2VydHMu
// SIG // ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0U0hBMkFzc3VyZWRJ
// SIG // RFRpbWVzdGFtcGluZ0NBLmNydDANBgkqhkiG9w0BAQsF
// SIG // AAOCAQEALoOhRAVKBOO5MlL62YHwGrv4CY0juT3YkqHm
// SIG // RhxKL256PGNuNxejGr9YI7JDnJSDTjkJsCzox+HizO3L
// SIG // eWvO3iMBR+2VVIHggHsSsa8Chqk6c2r++J/BjdEhjOQp
// SIG // gsOKC2AAAp0fR8SftApoU39aEKb4Iub4U5IxX9iCgy1t
// SIG // E0Kug8EQTqQk9Eec3g8icndcf0/pOZgrV5JE1+9uk9lD
// SIG // xwQzY1E3Vp5HBBHDo1hUIdjijlbXST9X/AqfI1579JSN
// SIG // 3Z0au996KqbSRaZVDI/2TIryls+JRtwxspGQo18zMGBV
// SIG // 9fxrMKyh7eRHTjOeZ2ootU3C7VuXgvjLqQhsUwm09zCC
// SIG // BTEwggQZoAMCAQICEAqhJdbWMht+QeQF2jaXwhUwDQYJ
// SIG // KoZIhvcNAQELBQAwZTELMAkGA1UEBhMCVVMxFTATBgNV
// SIG // BAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRp
// SIG // Z2ljZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNz
// SIG // dXJlZCBJRCBSb290IENBMB4XDTE2MDEwNzEyMDAwMFoX
// SIG // DTMxMDEwNzEyMDAwMFowcjELMAkGA1UEBhMCVVMxFTAT
// SIG // BgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3
// SIG // LmRpZ2ljZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQg
// SIG // U0hBMiBBc3N1cmVkIElEIFRpbWVzdGFtcGluZyBDQTCC
// SIG // ASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAL3Q
// SIG // Mu5LzY9/3am6gpnFOVQoV7YjSsQOB0UzURB90Pl9TWh+
// SIG // 57ag9I2ziOSXv2MhkJi/E7xX08PhfgjWahQAOPcuHjvu
// SIG // zKb2Mln+X2U/4Jvr40ZHBhpVfgsnfsCi9aDg3iI/Dv9+
// SIG // lfvzo7oiPhisEeTwmQNtO4V8CdPuXciaC1TjqAlxa+DP
// SIG // IhAPdc9xck4Krd9AOly3UeGheRTGTSQjMF287Dxgaqwv
// SIG // B8z98OpH2YhQXv1mblZhJymJhFHmgudGUP2UKiyn5HU+
// SIG // upgPhH+fMRTWrdXyZMt7HgXQhBlyF/EXBu89zdZN7wZC
// SIG // /aJTKk+FHcQdPK/P2qwQ9d2srOlW/5MCAwEAAaOCAc4w
// SIG // ggHKMB0GA1UdDgQWBBT0tuEgHf4prtLkYaWyoiWyyBc1
// SIG // bjAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823I
// SIG // DzASBgNVHRMBAf8ECDAGAQH/AgEAMA4GA1UdDwEB/wQE
// SIG // AwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB5BggrBgEF
// SIG // BQcBAQRtMGswJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3Nw
// SIG // LmRpZ2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3aHR0cDov
// SIG // L2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNz
// SIG // dXJlZElEUm9vdENBLmNydDCBgQYDVR0fBHoweDA6oDig
// SIG // NoY0aHR0cDovL2NybDQuZGlnaWNlcnQuY29tL0RpZ2lD
// SIG // ZXJ0QXNzdXJlZElEUm9vdENBLmNybDA6oDigNoY0aHR0
// SIG // cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNz
// SIG // dXJlZElEUm9vdENBLmNybDBQBgNVHSAESTBHMDgGCmCG
// SIG // SAGG/WwAAgQwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93
// SIG // d3cuZGlnaWNlcnQuY29tL0NQUzALBglghkgBhv1sBwEw
// SIG // DQYJKoZIhvcNAQELBQADggEBAHGVEulRh1Zpze/d2nyq
// SIG // Y3qzeM8GN0CE70uEv8rPAwL9xafDDiBCLK938ysfDCFa
// SIG // KrcFNB1qrpn4J6JmvwmqYN92pDqTD/iy0dh8GWLoXoIl
// SIG // HsS6HHssIeLWWywUNUMEaLLbdQLgcseY1jxk5R9IEBhf
// SIG // iThhTWJGJIdjjJFSLK8pieV4H9YLFKWA1xJHcLN11ZOF
// SIG // k362kmf7U2GJqPVrlsD0WGkNfMgBsbkodbeZY4UijGHK
// SIG // eZR+WfyMD+NvtQEmtmyl7odRIeRYYJu6DC0rbaLEfrvE
// SIG // JStHAgh8Sa4TtuF8QkIoxhhWz0E0tmZdtnR79VYzIi8i
// SIG // NrJLokqV2PWmjlIxggJNMIICSQIBATCBhjByMQswCQYD
// SIG // VQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkw
// SIG // FwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQD
// SIG // EyhEaWdpQ2VydCBTSEEyIEFzc3VyZWQgSUQgVGltZXN0
// SIG // YW1waW5nIENBAhAEzT+FaK52xhuw/nFgzKdtMA0GCWCG
// SIG // SAFlAwQCAQUAoIGYMBoGCSqGSIb3DQEJAzENBgsqhkiG
// SIG // 9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcNMjAwNDAyMDYz
// SIG // MzM1WjArBgsqhkiG9w0BCRACDDEcMBowGDAWBBQDJb1Q
// SIG // XtqWMC3CL0+gHkwovig0xTAvBgkqhkiG9w0BCQQxIgQg
// SIG // utlyr7g6CG7Ck6rmz2NsrZLjeKFoh69t4w1uJ4OdFQUw
// SIG // DQYJKoZIhvcNAQEBBQAEggEAGDadaj6pgOmxoz3Q/OeA
// SIG // JxUjPj7BJZSsAJOxPfz29tbAqPx+xAsv8MAniTf1F74i
// SIG // MjDJnQXmMapraZtPN+t/KQvp/fPbXnlYGk6mNcBisjrw
// SIG // PcRMKQMUDqCA8cEy+4NMmExeoTabO9QgnznvJefbVPYP
// SIG // +4QrTV0oHkYfeTi1xO4k6EApE8ZSxkVoLznLpoMiVBp6
// SIG // 5h6fy/hZ4QSS4FJcmSctiFlRRRoikPDAKUSskq2WZbMm
// SIG // s+BSNGxpQs44ouhf58U44aGH6CjSmbo/IKSXbwEEHOa3
// SIG // ZBlu87ym4qD18I7e0UF0H1vcDRzMQLqLEjwiGTZVLAR4
// SIG // XaIOB0Laknd9EQ==
// SIG // End signature block
